<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class FinanceiroController extends Controller
{
    public function index(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // Receitas (Negociações fechadas)
        $receitas = DB::table('negociacoes')
            ->whereBetween(DB::raw('DATE(data_criacao)'), [$start, $end])
            ->whereIn('status', ['fechado'])
            ->select(
                'id',
                'valor_total',
                'valor_fechado',
                'valor_pago',
                'investimento_parcial',
                'num_parcelas',
                'data_criacao',
                'especialista_id'
            )
            ->orderBy('data_criacao', 'desc')
            ->get();

        $totalReceita = $receitas->sum('valor_total');
        $totalEntradas = $receitas->sum('investimento_parcial');
        $totalPago = $receitas->sum('valor_pago');

        // Pagamentos (se houver tabela)
        $pagamentos = DB::table('pagamentos')
            ->whereBetween(DB::raw('DATE(data_pagamento)'), [$start, $end])
            ->get();

        $totalPagamentos = $pagamentos->sum('valor');

        // Comissões
        $comissoes = DB::table('comissoes')
            ->whereBetween(DB::raw('DATE(data_prevista)'), [$start, $end])
            ->get();

        $totalComissoes = $comissoes->sum('valor');

        // Receita por especialista
        $receitaPorEspecialista = $receitas->groupBy('especialista_id')->map(function($grupo) {
            return [
                'total' => $grupo->count(),
                'valor' => $grupo->sum('valor_total'),
            ];
        });

        // Receita por dia
        $receitaPorDia = $receitas->groupBy(function($item) {
            return Carbon::parse($item->data_criacao)->format('Y-m-d');
        })->map(function($grupo) {
            return $grupo->sum('valor_total');
        });

        return view('admin.financeiro.index', compact(
            'start',
            'end',
            'receitas',
            'totalReceita',
            'totalEntradas',
            'totalPago',
            'pagamentos',
            'totalPagamentos',
            'comissoes',
            'totalComissoes',
            'receitaPorEspecialista',
            'receitaPorDia'
        ));
    }

    public function exportar(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        $receitas = DB::table('negociacoes')
            ->join('leads', 'negociacoes.lead_id', '=', 'leads.id')
            ->join('usuarios', 'negociacoes.especialista_id', '=', 'usuarios.id')
            ->whereBetween(DB::raw('DATE(negociacoes.data_criacao)'), [$start, $end])
            ->whereIn('negociacoes.status', ['fechado'])
            ->select(
                'negociacoes.id',
                'leads.nome as lead_nome',
                'usuarios.nome as especialista_nome',
                'negociacoes.valor_total',
                'negociacoes.valor_fechado',
                'negociacoes.valor_pago',
                'negociacoes.investimento_parcial',
                'negociacoes.num_parcelas',
                'negociacoes.data_criacao'
            )
            ->get();

        $filename = 'financeiro_' . date('Y-m-d_His') . '.csv';
        $handle = fopen('php://output', 'w');

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        // BOM para UTF-8
        fprintf($handle, chr(0xEF).chr(0xBB).chr(0xBF));

        // Cabeçalho
        fputcsv($handle, ['ID', 'Lead', 'Especialista', 'Valor Total', 'Valor Fechado', 'Valor Pago', 'Investimento Parcial', 'Parcelas', 'Data']);

        // Dados
        foreach ($receitas as $receita) {
            fputcsv($handle, [
                $receita->id,
                $receita->lead_nome,
                $receita->especialista_nome,
                number_format($receita->valor_total ?? 0, 2, ',', '.'),
                number_format($receita->valor_fechado ?? 0, 2, ',', '.'),
                number_format($receita->valor_pago ?? 0, 2, ',', '.'),
                number_format($receita->investimento_parcial ?? 0, 2, ',', '.'),
                $receita->num_parcelas ?? 1,
                Carbon::parse($receita->data_criacao)->format('d/m/Y H:i'),
            ]);
        }

        fclose($handle);
        exit;
    }
}

