<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Usuario;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class GestaoController extends Controller
{
    public function index(Request $request)
    {
        // Estatísticas gerais
        $stats = DB::table('leads')
            ->selectRaw('
                COUNT(*) as total_leads,
                COUNT(CASE WHEN especialista_id IS NOT NULL THEN 1 END) as com_especialista,
                COUNT(CASE WHEN equipe_id IS NOT NULL AND especialista_id IS NULL THEN 1 END) as com_equipe,
                COUNT(CASE WHEN especialista_id IS NULL AND equipe_id IS NULL THEN 1 END) as aguardando
            ')
            ->first();

        // Buscar equipes e especialistas para os selects
        $equipes = Equipe::where('status', 'ativo')->orderBy('nome')->get();
        $especialistas = Usuario::where('role', 'especialista')
            ->where('status', 'ativo')
            ->orderBy('nome')
            ->get();

        // Filtros
        $busca = trim($request->input('busca', ''));
        $filtroTipo = $request->input('tipo', '');
        $filtroId = $request->input('id', '');
        $filtroNome = '';
        $leads = collect();

        // Busca por nome ou telefone
        if (!empty($busca)) {
            $leads = Lead::with(['especialista', 'equipe'])
                ->where(function($q) use ($busca) {
                    $q->where('nome', 'like', "%{$busca}%")
                      ->orWhere('telefone', 'like', "%{$busca}%");
                })
                ->orderBy('data_cadastro', 'desc')
                ->limit(50)
                ->get();
            $filtroNome = "Busca: \"$busca\"";
        }
        // Filtro por especialista/equipe
        elseif ($filtroTipo && $filtroId) {
            if ($filtroTipo === 'especialista') {
                $leads = Lead::with('especialista')
                    ->where('especialista_id', $filtroId)
                    ->orderBy('data_cadastro', 'desc')
                    ->get();
                
                $especialista = Usuario::find($filtroId);
                $filtroNome = "Especialista: " . ($especialista->nome ?? '');
            } else {
                $leads = Lead::with('equipe')
                    ->where('equipe_id', $filtroId)
                    ->orderBy('data_cadastro', 'desc')
                    ->get();
                
                $equipe = Equipe::find($filtroId);
                $filtroNome = "Equipe: " . ($equipe->nome ?? '');
            }
        }

        return view('admin.gestao.index', compact(
            'stats',
            'equipes',
            'especialistas',
            'leads',
            'busca',
            'filtroTipo',
            'filtroId',
            'filtroNome'
        ));
    }

    public function transferir(Request $request)
    {
        $request->validate([
            'lead_id' => 'required|exists:leads,id',
            'tipo_destino' => 'required|in:especialista,equipe',
            'destino_id' => 'required|integer',
        ]);

        $lead = Lead::findOrFail($request->lead_id);

        if ($request->tipo_destino === 'especialista') {
            $especialista = Usuario::where('id', $request->destino_id)
                ->where('role', 'especialista')
                ->firstOrFail();

            // Buscar equipe do especialista
            $equipeEspecialista = DB::table('equipe_especialistas')
                ->where('especialista_id', $especialista->id)
                ->where('status', 'ativo')
                ->first();

            $lead->update([
                'especialista_id' => $especialista->id,
                'equipe_id' => $equipeEspecialista->equipe_id ?? null,
                'data_distribuicao' => now(),
            ]);

            return back()->with('success', 'Lead transferido para especialista com sucesso!');
        } else {
            $equipe = Equipe::findOrFail($request->destino_id);

            $lead->update([
                'equipe_id' => $equipe->id,
                'especialista_id' => null,
                'data_distribuicao' => now(),
            ]);

            return back()->with('success', 'Lead transferido para equipe com sucesso!');
        }
    }
}

