<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class HorarioController extends Controller
{
    public function index()
    {
        $horarios = DB::table('horarios_funcionamento')
            ->orderBy('dia_semana')
            ->get();

        $excecoes = DB::table('horarios_excecoes')
            ->where('data', '>=', now()->format('Y-m-d'))
            ->orderBy('data')
            ->get();

        return view('admin.horarios.index', compact('horarios', 'excecoes'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'horarios' => 'required|array',
            'horarios.*.dia_semana' => 'required|integer|between:0,6',
            'horarios.*.hora_inicio' => 'required|date_format:H:i',
            'horarios.*.hora_fim' => 'required|date_format:H:i',
            'horarios.*.ativo' => 'required|boolean',
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->horarios as $horario) {
                DB::table('horarios_funcionamento')
                    ->updateOrInsert(
                        ['dia_semana' => $horario['dia_semana']],
                        [
                            'hora_inicio' => $horario['hora_inicio'],
                            'hora_fim' => $horario['hora_fim'],
                            'ativo' => $horario['ativo'],
                            'atualizado_em' => now(),
                        ]
                    );
            }

            DB::commit();

            return redirect()->route('admin.horarios.index')
                ->with('success', 'Horários atualizados com sucesso!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Erro ao atualizar horários: ' . $e->getMessage());
        }
    }

    public function criarExcecao(Request $request)
    {
        $request->validate([
            'data' => 'required|date|after:today',
            'motivo' => 'required|string|max:255',
            'fechado' => 'required|boolean',
            'hora_inicio' => 'nullable|date_format:H:i',
            'hora_fim' => 'nullable|date_format:H:i',
        ]);

        DB::table('horarios_excecoes')->insert([
            'data' => $request->data,
            'motivo' => $request->motivo,
            'fechado' => $request->fechado,
            'hora_inicio' => $request->hora_inicio,
            'hora_fim' => $request->hora_fim,
            'criado_em' => now(),
        ]);

        return redirect()->route('admin.horarios.index')
            ->with('success', 'Exceção criada com sucesso!');
    }

    public function excluirExcecao($id)
    {
        DB::table('horarios_excecoes')->where('id', $id)->delete();

        return response()->json([
            'success' => true,
            'message' => 'Exceção excluída com sucesso!',
        ]);
    }
}

