<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Usuario;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class LeadController extends Controller
{
    /**
     * Lista todos os leads com funcionalidades de distribuição
     */
    public function index(Request $request)
    {
        // Verificar se é requisição AJAX para atualizar apenas a lista
        if ($request->get('ajax') === 'lista' && $request->header('X-Requested-With') === 'XMLHttpRequest') {
            return $this->getLeadsAguardando();
        }

        // Estatísticas de leads
        $statsLeads = DB::table('leads')
            ->selectRaw('
                COUNT(*) as total_leads,
                COUNT(CASE WHEN etapa = "novo" AND especialista_id IS NULL THEN 1 END) as aguardando,
                COUNT(CASE WHEN etapa = "novo" AND equipe_id IS NOT NULL AND especialista_id IS NULL THEN 1 END) as aguardando_coordenador,
                COUNT(CASE WHEN etapa = "contato" THEN 1 END) as contatado,
                COUNT(CASE WHEN etapa = "demo" THEN 1 END) as em_negociacao,
                COUNT(CASE WHEN etapa = "fechamento" THEN 1 END) as aprovada,
                COUNT(CASE WHEN etapa = "perdido" THEN 1 END) as rejeitada
            ')
            ->first();

        // Leads aguardando distribuição
        $leadsAguardando = DB::table('leads as l')
            ->leftJoin('equipes as e', 'l.equipe_id', '=', 'e.id')
            ->leftJoin('usuarios as c', 'l.coordenador_id', '=', 'c.id')
            ->leftJoin('usuarios as esp', 'l.especialista_id', '=', 'esp.id')
            ->whereIn('l.etapa', ['novo'])
            ->whereNull('l.especialista_id')
            ->select(
                'l.*',
                'e.nome as equipe_nome',
                'c.nome as coordenador_nome',
                'esp.nome as especialista_nome'
            )
            ->orderBy('l.data_cadastro', 'desc')
            ->get();

        // Leads distribuídos
        $leadsDistribuidos = DB::table('leads as l')
            ->leftJoin('usuarios as esp', 'l.especialista_id', '=', 'esp.id')
            ->leftJoin('equipes as e', 'l.equipe_id', '=', 'e.id')
            ->whereNotNull('l.especialista_id')
            ->select(
                'l.*',
                'esp.nome as especialista_nome',
                'e.nome as equipe_nome'
            )
            ->orderBy('l.data_distribuicao', 'desc')
            ->orderBy('l.data_cadastro', 'desc')
            ->limit(100)
            ->get();

        // Equipes disponíveis
        $equipes = DB::table('equipes as e')
            ->leftJoin('equipe_especialistas as ee', function($join) {
                $join->on('e.id', '=', 'ee.equipe_id')
                     ->where('ee.status', '=', 'ativo');
            })
            ->leftJoin('leads as l', 'e.id', '=', 'l.equipe_id')
            ->where('e.status', 'ativo')
            ->select(
                'e.id',
                'e.nome',
                'e.descricao',
                'e.meta_mensal',
                'e.tipo',
                'e.coordenador_id',
                'e.status',
                DB::raw('COUNT(DISTINCT ee.especialista_id) as total_especialistas'),
                DB::raw('COUNT(DISTINCT l.id) as total_leads')
            )
            ->groupBy('e.id', 'e.nome', 'e.descricao', 'e.meta_mensal', 'e.tipo', 'e.coordenador_id', 'e.status')
            ->orderBy('e.nome')
            ->get();

        // Especialistas disponíveis
        $especialistas = DB::table('usuarios as u')
            ->join('usuario_categoria as uc', 'u.id', '=', 'uc.usuario_id')
            ->join('categorias_usuario as c', 'uc.categoria_id', '=', 'c.id')
            ->leftJoin('equipe_especialistas as ee', function($join) {
                $join->on('u.id', '=', 'ee.especialista_id')
                     ->where('ee.status', '=', 'ativo');
            })
            ->leftJoin('equipes as e', 'ee.equipe_id', '=', 'e.id')
            ->leftJoin('leads as l', 'u.id', '=', 'l.especialista_id')
            ->where('c.slug', 'especialista')
            ->where('u.status', 'ativo')
            ->select(
                'u.id',
                'u.nome',
                'u.email',
                'e.nome as equipe_nome',
                DB::raw('COUNT(DISTINCT l.id) as total_leads')
            )
            ->groupBy('u.id', 'u.nome', 'u.email', 'e.nome')
            ->orderBy('u.nome')
            ->get();

        // Configuração de distribuição automática
        $distribuicaoAutomatica = DB::table('configuracoes')
            ->where('chave', 'distribuicao_automatica')
            ->value('valor') ?? '0';

        return view('admin.leads.index', compact(
            'statsLeads',
            'leadsAguardando',
            'leadsDistribuidos',
            'equipes',
            'especialistas',
            'distribuicaoAutomatica'
        ));
    }

    /**
     * Retorna apenas a lista de leads aguardando (AJAX)
     */
    private function getLeadsAguardando()
    {
        $leadsAguardando = DB::table('leads as l')
            ->leftJoin('equipes as e', 'l.equipe_id', '=', 'e.id')
            ->leftJoin('usuarios as c', 'l.coordenador_id', '=', 'c.id')
            ->leftJoin('usuarios as esp', 'l.especialista_id', '=', 'esp.id')
            ->whereIn('l.etapa', ['novo'])
            ->whereNull('l.especialista_id')
            ->select(
                'l.*',
                'e.nome as equipe_nome',
                'c.nome as coordenador_nome',
                'esp.nome as especialista_nome'
            )
            ->orderBy('l.data_cadastro', 'desc')
            ->get();

        return response()->json([
            'html' => view('admin.leads.partials.leads-aguardando', compact('leadsAguardando'))->render()
        ]);
    }

    /**
     * Processa ações POST
     */
    public function store(Request $request)
    {
        $action = $request->input('action');

        switch ($action) {
            case 'criar_lead':
                return $this->criarLead($request);
            case 'distribuir_automatico':
                return $this->distribuirAutomatico($request);
            case 'distribuir_manual':
                return $this->distribuirManual($request);
            case 'distribuir_para_equipe':
                return $this->distribuirParaEquipe($request);
            case 'distribuir_para_especialista':
                return $this->distribuirParaEspecialista($request);
            default:
                return response()->json([
                    'success' => false,
                    'message' => 'Ação não reconhecida'
                ], 400);
        }
    }

    /**
     * Cria um novo lead
     */
    private function criarLead(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nome' => 'required|string|max:255',
            'email' => 'required|email|unique:leads,email',
            'telefone' => 'required|string|max:20',
            'area_interesse' => 'required|string|max:150',
            'profissao' => 'required|string|max:150',
            'horario_preferencia' => 'nullable|string|max:100',
        ], [
            'nome.required' => 'Nome é obrigatório.',
            'email.required' => 'Email é obrigatório.',
            'email.email' => 'Email inválido.',
            'email.unique' => 'Já existe um lead com este email.',
            'telefone.required' => 'Telefone é obrigatório.',
            'area_interesse.required' => 'Área de interesse é obrigatória.',
            'profissao.required' => 'Profissão é obrigatória.',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            $lead = Lead::create([
                'nome' => $request->nome,
                'email' => $request->email,
                'telefone' => $request->telefone,
                'area_interesse' => $request->area_interesse,
                'profissao' => $request->profissao,
                'horario_preferencia' => $request->horario_preferencia,
                'etapa' => 'novo',
                'status' => 'ativo',
                'data_cadastro' => now(),
            ]);

            // Distribuição automática se estiver ativa
            $distribuicaoAutomatica = DB::table('configuracoes')
                ->where('chave', 'distribuicao_automatica')
                ->value('valor') ?? '0';

            $distribuido = false;
            if ($distribuicaoAutomatica == '1') {
                $distribuido = $this->distribuirLeadAutomaticamente($lead->id, auth()->id());
            }

            $message = $distribuido 
                ? 'Lead criado e distribuído automaticamente!' 
                : 'Lead criado com sucesso!';

            return redirect()->route('admin.leads.index')
                ->with('success', $message);

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao criar lead: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Distribui leads automaticamente para equipes
     */
    private function distribuirAutomatico(Request $request)
    {
        $equipeIds = $request->input('equipe_ids', []);

        if (empty($equipeIds)) {
            return back()->withErrors(['error' => 'Selecione pelo menos uma equipe.']);
        }

        try {
            // Buscar leads aguardando
            $leadsAguardando = DB::table('leads')
                ->where('etapa', 'novo')
                ->whereNull('especialista_id')
                ->orderBy('data_cadastro', 'asc')
                ->pluck('id')
                ->toArray();

            // Distribuir leads entre as equipes selecionadas
            foreach ($leadsAguardando as $index => $leadId) {
                $equipeIndex = $index % count($equipeIds);
                $equipeId = $equipeIds[$equipeIndex];

                // Buscar coordenador da equipe
                $coordenadorId = DB::table('equipes')
                    ->where('id', $equipeId)
                    ->value('coordenador_id');

                if ($coordenadorId) {
                    // Atualizar lead para aguardando coordenação
                    DB::table('leads')
                        ->where('id', $leadId)
                        ->update([
                            'etapa' => 'novo',
                            'equipe_id' => $equipeId,
                            'coordenador_id' => $coordenadorId,
                            'data_distribuicao' => now(),
                        ]);
                }
            }

            return redirect()->route('admin.leads.index')
                ->with('success', 'Leads distribuídos automaticamente para as equipes selecionadas!');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao distribuir leads: ' . $e->getMessage()]);
        }
    }

    /**
     * Distribui lead manualmente para especialista
     */
    private function distribuirManual(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'lead_id' => 'required|exists:leads,id',
            'especialista_id' => 'required|exists:usuarios,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Dados inválidos'
            ], 400);
        }

        try {
            DB::table('leads')
                ->where('id', $request->lead_id)
                ->update([
                    'especialista_id' => $request->especialista_id,
                    'etapa' => 'contato',
                    'data_distribuicao' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Lead distribuído manualmente com sucesso!'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao distribuir lead: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Distribui lead para equipe (AJAX)
     */
    public function distribuirParaEquipe(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'lead_id' => 'required|exists:leads,id',
            'equipe_id' => 'required|exists:equipes,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'error' => 'Dados inválidos'
            ], 400);
        }

        try {
            // Buscar coordenador da equipe
            $coordenadorId = DB::table('equipes')
                ->where('id', $request->equipe_id)
                ->value('coordenador_id');

            DB::table('leads')
                ->where('id', $request->lead_id)
                ->update([
                    'equipe_id' => $request->equipe_id,
                    'coordenador_id' => $coordenadorId,
                    'etapa' => 'novo',
                    'data_distribuicao' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Lead distribuído para a equipe com sucesso!'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Erro ao distribuir lead: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Distribui lead para especialista (AJAX)
     */
    public function distribuirParaEspecialista(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'lead_id' => 'required|exists:leads,id',
            'especialista_id' => 'required|exists:usuarios,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'error' => 'Dados inválidos'
            ], 400);
        }

        try {
            // Buscar equipe do especialista
            $equipeEspecialista = DB::table('equipe_especialistas')
                ->where('especialista_id', $request->especialista_id)
                ->where('status', 'ativo')
                ->first();

            DB::table('leads')
                ->where('id', $request->lead_id)
                ->update([
                    'especialista_id' => $request->especialista_id,
                    'equipe_id' => $equipeEspecialista->equipe_id ?? null,
                    'etapa' => 'contato',
                    'data_distribuicao' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Lead distribuído para o especialista com sucesso!'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Erro ao distribuir lead: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle distribuição automática
     */
    public function toggleDistribuicaoAutomatica(Request $request)
    {
        $ativo = $request->input('ativo', 0);

        try {
            DB::table('configuracoes')
                ->updateOrInsert(
                    ['chave' => 'distribuicao_automatica'],
                    ['valor' => $ativo, 'atualizado_em' => now()]
                );

            return response()->json([
                'success' => true,
                'message' => $ativo ? 'Distribuição automática ativada!' : 'Distribuição automática desativada!'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar configuração: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Distribui lead automaticamente (função auxiliar)
     */
    private function distribuirLeadAutomaticamente($leadId, $usuarioId)
    {
        try {
            // Buscar especialista com menos leads
            $especialista = DB::table('usuarios as u')
                ->join('usuario_categoria as uc', 'u.id', '=', 'uc.usuario_id')
                ->join('categorias_usuario as c', 'uc.categoria_id', '=', 'c.id')
                ->leftJoin('leads as l', 'u.id', '=', 'l.especialista_id')
                ->where('c.slug', 'especialista')
                ->where('u.status', 'ativo')
                ->where('uc.ativo', 1)
                ->select(
                    'u.id',
                    DB::raw('COUNT(l.id) as total_leads')
                )
                ->groupBy('u.id')
                ->orderBy('total_leads', 'asc')
                ->first();

            if ($especialista) {
                // Buscar equipe do especialista
                $equipeEspecialista = DB::table('equipe_especialistas')
                    ->where('especialista_id', $especialista->id)
                    ->where('status', 'ativo')
                    ->first();

                DB::table('leads')
                    ->where('id', $leadId)
                    ->update([
                        'especialista_id' => $especialista->id,
                        'equipe_id' => $equipeEspecialista->equipe_id ?? null,
                        'etapa' => 'contato',
                        'data_distribuicao' => now(),
                    ]);

                return true;
            }

            return false;

        } catch (\Exception $e) {
            \Log::error('Erro ao distribuir lead automaticamente: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Exibe detalhes de um lead
     */
    public function show($id)
    {
        $lead = Lead::with(['especialista', 'equipe', 'atendimentos'])->findOrFail($id);

        // Buscar negociações do lead
        $negociacoes = DB::table('negociacoes')
            ->where('lead_id', $id)
            ->orderBy('data_criacao', 'desc')
            ->get();

        return view('admin.leads.show', compact('lead', 'negociacoes'));
    }
}
