<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class MarketingController extends Controller
{
    public function index()
    {
        $links = DB::table('links_marketing')
            ->orderBy('data_criacao', 'desc')
            ->paginate(50);

        $clicks = DB::table('clicks_marketing')
            ->select('link_id', DB::raw('COUNT(*) as total_clicks'))
            ->groupBy('link_id')
            ->pluck('total_clicks', 'link_id');

        return view('admin.marketing.index', compact('links', 'clicks'));
    }

    public function create()
    {
        return view('admin.marketing.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'nome' => 'required|string|max:255',
            'utm_source' => 'required|string|max:100',
            'utm_medium' => 'required|string|max:100',
            'utm_campaign' => 'required|string|max:100',
            'utm_content' => 'nullable|string|max:100',
            'utm_term' => 'nullable|string|max:100',
        ]);

        // Gerar código único
        $codigo = strtoupper(substr(md5(uniqid()), 0, 8));

        // URL de destino fixa
        $urlDestino = 'https://vencerleitura.com.br/agendamento';

        // Construir URL completa com parâmetros UTM
        $urlParams = array_filter([
            'utm_source' => $request->utm_source,
            'utm_medium' => $request->utm_medium,
            'utm_campaign' => $request->utm_campaign,
            'utm_content' => $request->utm_content,
            'utm_term' => $request->utm_term,
            'ref' => $codigo,
        ]);

        $urlCompleta = $urlDestino . '?' . http_build_query($urlParams);

        // Descrição com os parâmetros UTM para referência
        $descricao = "UTM: source={$request->utm_source}, medium={$request->utm_medium}, campaign={$request->utm_campaign}";
        if ($request->utm_content) {
            $descricao .= ", content={$request->utm_content}";
        }
        if ($request->utm_term) {
            $descricao .= ", term={$request->utm_term}";
        }

        DB::table('links_marketing')->insert([
            'nome' => $request->nome,
            'codigo' => $codigo,
            'url_completa' => $urlCompleta,
            'descricao' => $descricao,
            'ativo' => 1,
            'criado_por' => Auth::id(),
            'data_criacao' => now(),
        ]);

        return redirect()->route('admin.marketing.index')
            ->with('success', "Link criado com sucesso! Código: {$codigo} | URL: " . url('/l/' . $codigo));
    }

    public function show($id)
    {
        $link = DB::table('links_marketing')->where('id', $id)->first();
        
        if (!$link) {
            abort(404);
        }

        // Estatísticas do link
        $clicks = DB::table('clicks_marketing')
            ->where('link_id', $id)
            ->get();

        $totalClicks = $clicks->count();

        // Clicks por dia
        $clicksPorDia = DB::table('clicks_marketing')
            ->where('link_id', $id)
            ->select(DB::raw('DATE(data_click) as data'), DB::raw('COUNT(*) as total'))
            ->groupBy('data')
            ->orderBy('data', 'desc')
            ->limit(30)
            ->get();

        // Extrair UTM da descrição para buscar leads
        $utmSource = null;
        $utmCampaign = null;
        if ($link->descricao) {
            if (preg_match('/source=([^,]+)/', $link->descricao, $matches)) {
                $utmSource = $matches[1];
            }
            if (preg_match('/campaign=([^,]+)/', $link->descricao, $matches)) {
                $utmCampaign = $matches[1];
            }
        }

        // Leads gerados (buscar por código do link na origem)
        $leadsGerados = DB::table('leads')
            ->where('link_marketing_id', $id)
            ->count();

        return view('admin.marketing.show', compact('link', 'totalClicks', 'clicksPorDia', 'leadsGerados'));
    }

    public function toggleStatus($id)
    {
        $link = DB::table('links_marketing')->where('id', $id)->first();

        if (!$link) {
            return response()->json(['success' => false, 'message' => 'Link não encontrado'], 404);
        }

        DB::table('links_marketing')
            ->where('id', $id)
            ->update(['ativo' => !$link->ativo]);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado!',
            'novo_status' => !$link->ativo,
        ]);
    }
}

