<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Usuario;
use App\Models\Aluno;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class UsuarioController extends Controller
{
    /**
     * Lista todos os usuários
     */
    public function index(Request $request)
    {
        $query = Usuario::query()
            ->leftJoin('usuario_categoria', function($join) {
                $join->on('usuarios.id', '=', 'usuario_categoria.usuario_id')
                     ->where('usuario_categoria.ativo', '=', 1);
            })
            ->leftJoin('categorias_usuario', 'usuario_categoria.categoria_id', '=', 'categorias_usuario.id')
            ->select('usuarios.*', 'categorias_usuario.nome as categoria_nome', 'categorias_usuario.slug as categoria_slug')
            ->distinct();

        // Filtros
        if ($request->filled('categoria_id')) {
            $query->where('categorias_usuario.id', $request->categoria_id);
        }

        if ($request->filled('status')) {
            $query->where('usuarios.status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('usuarios.nome', 'like', "%{$search}%")
                  ->orWhere('usuarios.email', 'like', "%{$search}%")
                  ->orWhere('usuarios.telefone', 'like', "%{$search}%");
            });
        }

        $usuarios = $query->orderBy('usuarios.nome')->paginate(50);

        // Buscar categorias para o filtro
        $categorias = DB::table('categorias_usuario')
            ->where('ativo', 1)
            ->orderBy('nome')
            ->get();

        return view('admin.usuarios.index', compact('usuarios', 'categorias'));
    }

    /**
     * Exibe formulário de criação
     */
    public function create()
    {
        $categorias = DB::table('categorias_usuario')
            ->where('ativo', 1)
            ->orderBy('nome')
            ->get();
        
        return view('admin.usuarios.create', compact('categorias'));
    }

    /**
     * Cria um novo usuário
     */
    public function store(Request $request)
    {
        $request->validate([
            'nome' => 'required|string|max:255',
            'email' => 'required|email|unique:usuarios,email',
            'senha' => 'required|min:6|confirmed',
            'telefone' => 'nullable|string|max:20',
            'categoria_id' => 'required|exists:categorias_usuario,id',
            'status' => 'required|in:ativo,inativo',
        ], [
            'nome.required' => 'O nome é obrigatório.',
            'email.required' => 'O email é obrigatório.',
            'email.email' => 'Informe um email válido.',
            'email.unique' => 'Este email já está cadastrado.',
            'senha.required' => 'A senha é obrigatória.',
            'senha.min' => 'A senha deve ter no mínimo 6 caracteres.',
            'senha.confirmed' => 'As senhas não conferem.',
            'categoria_id.required' => 'Selecione uma categoria.',
            'categoria_id.exists' => 'Categoria selecionada não existe.',
        ]);

        // Buscar a categoria para obter o slug (que será usado como role)
        $categoria = DB::table('categorias_usuario')
            ->where('id', $request->categoria_id)
            ->first();

        if (!$categoria) {
            return back()->withErrors(['categoria_id' => 'Categoria não encontrada.'])->withInput();
        }

        // Mapear slug da categoria para valores válidos do ENUM role
        $rolesValidos = ['admin', 'especialista', 'aluno', 'coordenador', 'professor', 'secretaria', 'gestao_pedagogica', 'marketing'];
        $role = in_array($categoria->slug, $rolesValidos) ? $categoria->slug : null;

        if (!$role) {
            return back()->withErrors(['categoria_id' => 'A categoria selecionada não possui um role válido. Entre em contato com o administrador.'])->withInput();
        }

        $usuario = Usuario::create([
            'nome' => $request->nome,
            'email' => $request->email,
            'senha' => Hash::make($request->senha),
            'telefone' => $request->telefone,
            'role' => $role,
            'status' => $request->status,
            'data_cadastro' => now(),
        ]);

        // Vincular usuário à categoria na tabela pivot
        DB::table('usuario_categoria')->insert([
            'usuario_id' => $usuario->id,
            'categoria_id' => $request->categoria_id,
            'ativo' => 1,
            'data_vinculo' => now(),
        ]);

        // Criar registros relacionados baseado na categoria
        if ($categoria->slug === 'aluno') {
            // Criar registro na tabela alunos se necessário
            DB::table('alunos')->insert([
                'usuario_id' => $usuario->id,
                'nome' => $usuario->nome,
                'email' => $usuario->email,
                'status' => 'ativo',
                'data_cadastro' => now(),
            ]);
        }

        return redirect()->route('admin.usuarios.index')
            ->with('success', 'Usuário criado com sucesso!');
    }

    /**
     * Exibe detalhes de um usuário
     */
    public function show($id)
    {
        $usuario = Usuario::findOrFail($id);

        // Buscar categoria do usuário
        $categoria = DB::table('usuario_categoria')
            ->join('categorias_usuario', 'usuario_categoria.categoria_id', '=', 'categorias_usuario.id')
            ->where('usuario_categoria.usuario_id', $id)
            ->where('usuario_categoria.ativo', 1)
            ->select('categorias_usuario.*')
            ->first();

        // Buscar dados adicionais se for aluno
        $dadosAluno = null;
        if ($usuario->role === 'aluno') {
            $dadosAluno = DB::table('alunos')
                ->where('usuario_id', $id)
                ->first();
        }

        return view('admin.usuarios.show', compact('usuario', 'dadosAluno', 'categoria'));
    }

    /**
     * Exibe formulário de edição
     */
    public function edit($id)
    {
        $usuario = Usuario::findOrFail($id);
        
        // Buscar categoria atual do usuário
        $categoriaAtual = DB::table('usuario_categoria')
            ->where('usuario_id', $id)
            ->where('ativo', 1)
            ->first();
        
        $categorias = DB::table('categorias_usuario')
            ->where('ativo', 1)
            ->orderBy('nome')
            ->get();
        
        return view('admin.usuarios.edit', compact('usuario', 'categorias', 'categoriaAtual'));
    }

    /**
     * Atualiza um usuário
     */
    public function update(Request $request, $id)
    {
        $usuario = Usuario::findOrFail($id);

        $request->validate([
            'nome' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('usuarios')->ignore($id)],
            'senha' => 'nullable|min:6|confirmed',
            'telefone' => 'nullable|string|max:20',
            'categoria_id' => 'required|exists:categorias_usuario,id',
            'status' => 'required|in:ativo,inativo',
        ]);

        // Buscar a categoria para obter o slug (que será usado como role)
        $categoria = DB::table('categorias_usuario')
            ->where('id', $request->categoria_id)
            ->first();

        if (!$categoria) {
            return back()->withErrors(['categoria_id' => 'Categoria não encontrada.'])->withInput();
        }

        // Mapear slug da categoria para valores válidos do ENUM role
        $rolesValidos = ['admin', 'especialista', 'aluno', 'coordenador', 'professor', 'secretaria', 'gestao_pedagogica', 'marketing'];
        $role = in_array($categoria->slug, $rolesValidos) ? $categoria->slug : null;

        if (!$role) {
            return back()->withErrors(['categoria_id' => 'A categoria selecionada não possui um role válido. Entre em contato com o administrador.'])->withInput();
        }

        $data = [
            'nome' => $request->nome,
            'email' => $request->email,
            'telefone' => $request->telefone,
            'role' => $role,
            'status' => $request->status,
        ];

        // Atualizar senha apenas se foi informada
        if ($request->filled('senha')) {
            $data['senha'] = Hash::make($request->senha);
        }

        $usuario->update($data);

        // Atualizar vínculo com categoria
        // Desativar vínculo anterior
        DB::table('usuario_categoria')
            ->where('usuario_id', $id)
            ->update(['ativo' => 0]);

        // Verificar se já existe vínculo com a nova categoria
        $vinculoExistente = DB::table('usuario_categoria')
            ->where('usuario_id', $id)
            ->where('categoria_id', $request->categoria_id)
            ->first();

        if ($vinculoExistente) {
            // Reativar vínculo existente
            DB::table('usuario_categoria')
                ->where('usuario_id', $id)
                ->where('categoria_id', $request->categoria_id)
                ->update([
                    'ativo' => 1,
                    'data_vinculo' => now(),
                ]);
        } else {
            // Criar novo vínculo
            DB::table('usuario_categoria')->insert([
                'usuario_id' => $id,
                'categoria_id' => $request->categoria_id,
                'ativo' => 1,
                'data_vinculo' => now(),
            ]);
        }

        return redirect()->route('admin.usuarios.index')
            ->with('success', 'Usuário atualizado com sucesso!');
    }

    /**
     * Remove um usuário
     */
    public function destroy($id)
    {
        $usuario = Usuario::findOrFail($id);

        // Não permitir deletar o próprio usuário
        if ($usuario->id === auth()->id()) {
            return back()->with('error', 'Você não pode deletar seu próprio usuário.');
        }

        // Desativar ao invés de deletar
        $usuario->update(['status' => 'inativo']);

        return redirect()->route('admin.usuarios.index')
            ->with('success', 'Usuário desativado com sucesso!');
    }
}

