<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class LoginController extends Controller
{
    /**
     * Exibe o formulário de login
     */
    public function showLoginForm()
    {
        // Se já estiver logado, redirecionar para o dashboard apropriado
        if (Auth::check()) {
            return $this->redirectToDashboard(Auth::user());
        }

        return view('auth.login');
    }

    /**
     * Processa o login
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'senha' => 'required',
        ], [
            'email.required' => 'Por favor, informe seu email.',
            'email.email' => 'Por favor, informe um email válido.',
            'senha.required' => 'Por favor, informe sua senha.',
        ]);

        $email = $request->input('email');
        $senha = $request->input('senha');

        // Buscar usuário
        $usuario = Usuario::where('email', $email)
            ->where('status', 'ativo')
            ->first();

        if (!$usuario) {
            return back()->withErrors([
                'email' => 'Email ou senha incorretos. Verifique suas credenciais.',
            ])->withInput($request->only('email'));
        }

        // Verificar senha
        if (!Hash::check($senha, $usuario->senha)) {
            return back()->withErrors([
                'email' => 'Email ou senha incorretos. Verifique suas credenciais.',
            ])->withInput($request->only('email'));
        }

        // Verificar se tem role definida
        if (empty($usuario->role)) {
            return back()->withErrors([
                'email' => 'Usuário sem role definida. Entre em contato com o administrador.',
            ])->withInput($request->only('email'));
        }

        // Fazer login
        Auth::login($usuario, false);

        // Atualizar último login
        $usuario->update(['ultimo_login' => now()]);

        // Redirecionar para o dashboard apropriado
        return $this->redirectToDashboard($usuario);
    }

    /**
     * Redireciona para o dashboard baseado na role do usuário
     */
    protected function redirectToDashboard(Usuario $usuario)
    {
        return match($usuario->role) {
            'admin' => redirect()->route('admin.dashboard'),
            'especialista' => redirect()->route('especialista.dashboard'),
            'coordenador' => redirect()->route('coordenador.dashboard'),
            'secretaria' => redirect()->route('secretaria.dashboard'),
            'professor' => redirect()->route('professor.dashboard'),
            'aluno' => redirect()->route('aluno.dashboard'),
            'marketing' => redirect()->route('admin.dashboard'), // Por enquanto redireciona para admin
            default => redirect()->route('home'),
        };
    }

    /**
     * Faz logout do usuário
     */
    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('success', 'Você saiu do sistema com sucesso.');
    }
}

