<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AtendimentoController extends Controller
{
    /**
     * Kanban de Atendimentos - Visão Coordenador (apenas leads das suas equipes)
     */
    public function index(Request $request)
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Filtros
        $filtroBusca = $request->input('busca', '');
        $filtroAtribuido = $request->input('atribuido', '');

        // Query base - apenas leads das equipes do coordenador
        $query = DB::table('leads as l')
            ->leftJoin('usuarios as esp', 'l.especialista_id', '=', 'esp.id')
            ->leftJoin('equipes as eq', 'l.equipe_id', '=', 'eq.id')
            ->whereIn('l.equipe_id', $equipeIds)
            ->select(
                'l.id',
                'l.nome',
                'l.email',
                'l.telefone',
                'l.profissao',
                'l.etapa',
                'l.status',
                'l.especialista_id',
                'l.equipe_id',
                'l.data_cadastro',
                'l.data_distribuicao',
                'l.data_atualizacao',
                'l.observacoes',
                'l.horario_preferencia',
                'l.area_interesse',
                'l.data_agendamento',
                'l.horario_agendamento',
                'esp.nome as especialista_nome',
                'eq.nome as equipe_nome',
                DB::raw('CASE WHEN l.especialista_id IS NOT NULL THEN 1 ELSE 0 END as distribuido'),
                DB::raw("CASE 
                    WHEN l.etapa = 'novo' THEN 'aguardando'
                    WHEN l.etapa = 'contato' THEN 'contatado'
                    WHEN l.etapa = 'demo' THEN 'negociacao'
                    WHEN l.etapa = 'fechamento' THEN 'aprovada'
                    WHEN l.etapa = 'perdido' THEN 'rejeitada'
                    ELSE 'aguardando'
                END as status_kanban")
            );

        // Aplicar filtros
        if (!empty($filtroBusca)) {
            $query->where(function($q) use ($filtroBusca) {
                $q->where('l.nome', 'like', "%{$filtroBusca}%")
                  ->orWhere('l.email', 'like', "%{$filtroBusca}%")
                  ->orWhere('l.telefone', 'like', "%{$filtroBusca}%");
            });
        }

        if (!empty($filtroAtribuido)) {
            if ($filtroAtribuido === 'nao_atribuido') {
                $query->whereNull('l.especialista_id');
            } elseif (strpos($filtroAtribuido, 'esp_') === 0) {
                $espId = substr($filtroAtribuido, 4);
                // Verificar se o especialista pertence às equipes do coordenador
                $especialistaEquipe = DB::table('equipe_especialistas')
                    ->where('especialista_id', $espId)
                    ->whereIn('equipe_id', $equipeIds)
                    ->where('status', 'ativo')
                    ->exists();
                
                if ($especialistaEquipe) {
                    $query->where('l.especialista_id', $espId);
                }
            }
        }

        $leads = $query->orderByRaw("
            CASE 
                WHEN l.etapa = 'novo' THEN 1
                WHEN l.etapa = 'contato' THEN 2
                WHEN l.etapa = 'demo' THEN 3
                WHEN l.etapa = 'fechamento' THEN 4
                ELSE 5
            END
        ")
        ->orderBy('l.data_atualizacao', 'desc')
        ->orderBy('l.data_cadastro', 'desc')
        ->get();

        // Especialistas das equipes do coordenador para filtro
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->leftJoin('leads as l', function($join) use ($equipeIds) {
                $join->on('u.id', '=', 'l.especialista_id')
                     ->whereIn('l.equipe_id', $equipeIds);
            })
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome', DB::raw('COUNT(l.id) as total_leads'))
            ->groupBy('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();

        // Estatísticas apenas das equipes do coordenador
        $stats = DB::table('leads')
            ->whereIn('equipe_id', $equipeIds)
            ->select(
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(CASE WHEN etapa = "novo" THEN 1 ELSE 0 END) as aguardando'),
                DB::raw('SUM(CASE WHEN etapa = "contato" THEN 1 ELSE 0 END) as contatado'),
                DB::raw('SUM(CASE WHEN etapa = "demo" THEN 1 ELSE 0 END) as negociacao'),
                DB::raw('SUM(CASE WHEN etapa = "fechamento" THEN 1 ELSE 0 END) as aprovada'),
                DB::raw('SUM(CASE WHEN etapa = "perdido" THEN 1 ELSE 0 END) as rejeitada'),
                DB::raw('SUM(CASE WHEN especialista_id IS NOT NULL THEN 1 ELSE 0 END) as distribuidos')
            )
            ->first();

        return view('coordenador.atendimentos.index', compact(
            'leads', 
            'especialistas', 
            'equipes',
            'filtroBusca', 
            'filtroAtribuido', 
            'stats'
        ));
    }

    /**
     * Atualizar status do lead via AJAX (Drag and Drop)
     */
    public function atualizarStatus(Request $request, $id)
    {
        $coordenador = Auth::user();
        
        // Verificar se o lead pertence às equipes do coordenador
        $equipeIds = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->pluck('id')
            ->toArray();
        
        $lead = DB::table('leads')
            ->where('id', $id)
            ->whereIn('equipe_id', $equipeIds)
            ->first();
        
        if (!$lead) {
            return response()->json([
                'success' => false,
                'message' => 'Lead não encontrado ou não pertence às suas equipes',
            ], 404);
        }

        $request->validate([
            'status' => 'required|in:aguardando,contatado,negociacao,aprovada,rejeitada',
        ]);

        // Mapear status do kanban para etapa do banco
        $etapaMap = [
            'aguardando' => 'novo',
            'contatado' => 'contato',
            'negociacao' => 'demo',
            'aprovada' => 'fechamento',
            'rejeitada' => 'perdido',
        ];

        $etapa = $etapaMap[$request->status] ?? 'novo';

        DB::table('leads')
            ->where('id', $id)
            ->update([
                'etapa' => $etapa,
                'data_atualizacao' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado!',
            'lead_nome' => $lead->nome,
        ]);
    }
}

