<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class FeedbackController extends Controller
{
    /**
     * Exibe a página de feedback
     */
    public function index()
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Verificar se a tabela feedback_coordenador existe, se não, usar feedback_modulos
        $tableExists = DB::select("SHOW TABLES LIKE 'feedback_coordenador'");
        
        if (empty($tableExists)) {
            // Criar tabela se não existir
            DB::statement("
                CREATE TABLE IF NOT EXISTS feedback_coordenador (
                    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                    coordenador_id BIGINT UNSIGNED NOT NULL,
                    especialista_id BIGINT UNSIGNED NULL,
                    tipo VARCHAR(50) NOT NULL,
                    categoria VARCHAR(50) NOT NULL,
                    titulo VARCHAR(255) NOT NULL,
                    descricao TEXT NOT NULL,
                    prioridade VARCHAR(20) NOT NULL,
                    status VARCHAR(20) NOT NULL DEFAULT 'pendente',
                    observacao TEXT NULL,
                    data_criacao DATETIME NOT NULL,
                    data_atualizacao DATETIME NULL,
                    INDEX idx_coordenador (coordenador_id),
                    INDEX idx_especialista (especialista_id),
                    INDEX idx_status (status)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
        }
        
        // Obter feedbacks do coordenador
        $feedbacks = DB::table('feedback_coordenador as fc')
            ->leftJoin('usuarios as u', 'fc.especialista_id', '=', 'u.id')
            ->where('fc.coordenador_id', $coordenador->id)
            ->select(
                'fc.*',
                'u.nome as especialista_nome'
            )
            ->orderBy('fc.data_criacao', 'desc')
            ->get();
        
        // Estatísticas dos feedbacks
        $totalFeedbacks = $feedbacks->count();
        $feedbacksPendentes = $feedbacks->where('status', 'pendente')->count();
        $feedbacksEmAnalise = $feedbacks->where('status', 'em_analise')->count();
        $feedbacksResolvidos = $feedbacks->where('status', 'resolvido')->count();
        
        // Especialistas da equipe para o formulário
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();
        
        return view('coordenador.feedback.index', compact(
            'equipes',
            'feedbacks',
            'totalFeedbacks',
            'feedbacksPendentes',
            'feedbacksEmAnalise',
            'feedbacksResolvidos',
            'especialistas'
        ));
    }
    
    /**
     * Criar novo feedback
     */
    public function store(Request $request)
    {
        $request->validate([
            'tipo' => 'required|string|in:venda,especialista,equipe,sistema,outro',
            'categoria' => 'required|string|in:melhoria,problema,sugestao,elogio,outro',
            'titulo' => 'required|string|max:255',
            'descricao' => 'required|string',
            'prioridade' => 'required|string|in:baixa,media,alta,urgente',
            'especialista_id' => 'nullable|exists:usuarios,id',
        ]);
        
        $coordenador = Auth::user();
        
        // Verificar se especialista pertence às equipes do coordenador (se fornecido)
        if ($request->especialista_id) {
            $especialista = DB::table('equipe_especialistas as ee')
                ->join('equipes as e', 'ee.equipe_id', '=', 'e.id')
                ->where('ee.especialista_id', $request->especialista_id)
                ->where('e.coordenador_id', $coordenador->id)
                ->where('ee.status', 'ativo')
                ->exists();
            
            if (!$especialista) {
                return back()->withErrors(['error' => 'Especialista não pertence às suas equipes.']);
            }
        }
        
        // Criar feedback
        DB::table('feedback_coordenador')->insert([
            'coordenador_id' => $coordenador->id,
            'especialista_id' => $request->especialista_id,
            'tipo' => $request->tipo,
            'categoria' => $request->categoria,
            'titulo' => $request->titulo,
            'descricao' => $request->descricao,
            'prioridade' => $request->prioridade,
            'status' => 'pendente',
            'data_criacao' => now(),
        ]);
        
        return back()->with('success', 'Feedback criado com sucesso!');
    }
    
    /**
     * Atualizar status do feedback
     */
    public function updateStatus(Request $request)
    {
        $request->validate([
            'feedback_id' => 'required|exists:feedback_coordenador,id',
            'novo_status' => 'required|string|in:pendente,em_analise,resolvido,fechado',
            'observacao' => 'nullable|string',
        ]);
        
        $coordenador = Auth::user();
        
        // Verificar se o feedback pertence ao coordenador
        $feedback = DB::table('feedback_coordenador')
            ->where('id', $request->feedback_id)
            ->where('coordenador_id', $coordenador->id)
            ->first();
        
        if (!$feedback) {
            return back()->withErrors(['error' => 'Feedback não encontrado ou não pertence a você.']);
        }
        
        // Atualizar status
        DB::table('feedback_coordenador')
            ->where('id', $request->feedback_id)
            ->update([
                'status' => $request->novo_status,
                'observacao' => $request->observacao,
                'data_atualizacao' => now(),
            ]);
        
        return back()->with('success', 'Status do feedback atualizado com sucesso!');
    }
}


