<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Equipe;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class LeadController extends Controller
{
    /**
     * Exibe a página de gestão de leads
     */
    public function index()
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Estatísticas de leads das equipes
        $statsLeads = DB::table('leads')
            ->whereIn('equipe_id', $equipeIds)
            ->selectRaw('
                COUNT(*) as total_leads,
                SUM(CASE WHEN etapa = "novo" THEN 1 ELSE 0 END) as aguardando_coordenador,
                SUM(CASE WHEN etapa = "contato" THEN 1 ELSE 0 END) as contatado,
                SUM(CASE WHEN etapa = "demo" THEN 1 ELSE 0 END) as em_negociacao,
                SUM(CASE WHEN etapa = "fechamento" THEN 1 ELSE 0 END) as aprovada,
                SUM(CASE WHEN etapa = "perdido" THEN 1 ELSE 0 END) as rejeitada
            ')
            ->first();
        
        // Leads aguardando distribuição (etapa = novo e pertencem às equipes do coordenador)
        $leadsAguardando = DB::table('leads as l')
            ->leftJoin('usuarios as esp', 'l.especialista_id', '=', 'esp.id')
            ->whereIn('l.equipe_id', $equipeIds)
            ->where('l.etapa', 'novo')
            ->select('l.*', 'esp.nome as especialista_nome')
            ->orderBy('l.data_cadastro', 'desc')
            ->get();
        
        // Especialistas das equipes
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->join('equipes as e', 'ee.equipe_id', '=', 'e.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select(
                'u.id',
                'u.nome',
                'u.email',
                'ee.data_entrada',
                DB::raw('(SELECT COUNT(*) FROM leads WHERE especialista_id = u.id) as total_leads'),
                DB::raw('(SELECT COUNT(*) FROM leads WHERE especialista_id = u.id AND etapa = "contato") as leads_contatado'),
                DB::raw('(SELECT COUNT(*) FROM leads WHERE especialista_id = u.id AND etapa = "demo") as leads_negociacao'),
                DB::raw('(SELECT COUNT(*) FROM leads WHERE especialista_id = u.id AND etapa = "fechamento") as leads_aprovada')
            )
            ->orderBy('u.nome')
            ->get();
        
        return view('coordenador.leads.index', compact(
            'equipes',
            'statsLeads',
            'leadsAguardando',
            'especialistas'
        ));
    }
    
    /**
     * Distribuir leads automaticamente
     */
    public function distribuirAutomatico(Request $request)
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->pluck('id')
            ->toArray();
        
        if (empty($equipes)) {
            return back()->withErrors(['error' => 'Você não possui equipes associadas.']);
        }
        
        // Buscar leads aguardando coordenação (etapa = novo e pertencem às equipes do coordenador)
        $leadsAguardando = DB::table('leads')
            ->whereIn('equipe_id', $equipes)
            ->where('etapa', 'novo')
            ->whereNull('especialista_id')
            ->orderBy('data_cadastro', 'asc')
            ->pluck('id')
            ->toArray();
        
        // Buscar especialistas das equipes
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipes)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->orderBy('ee.data_entrada', 'asc')
            ->pluck('u.id')
            ->toArray();
        
        if (empty($especialistas)) {
            return back()->withErrors(['error' => 'Não há especialistas disponíveis nas suas equipes.']);
        }
        
        if (empty($leadsAguardando)) {
            return back()->with('warning', 'Não há leads aguardando distribuição.');
        }
        
        // Distribuir leads entre especialistas
        $distribuidos = 0;
        foreach ($leadsAguardando as $index => $leadId) {
            $especialistaIndex = $index % count($especialistas);
            $especialistaId = $especialistas[$especialistaIndex];
            
            DB::table('leads')
                ->where('id', $leadId)
                ->update([
                    'especialista_id' => $especialistaId,
                    'etapa' => 'contato',
                    'data_distribuicao' => now(),
                ]);
            
            $distribuidos++;
        }
        
        return back()->with('success', "{$distribuidos} lead(s) distribuído(s) automaticamente para especialistas da equipe!");
    }
    
    /**
     * Distribuir lead manualmente
     */
    public function distribuirManual(Request $request)
    {
        $request->validate([
            'lead_id' => 'required|exists:leads,id',
            'especialista_id' => 'required|exists:usuarios,id',
        ]);
        
        $coordenador = Auth::user();
        
        // Verificar se o lead pertence às equipes do coordenador
        $equipeIds = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->pluck('id')
            ->toArray();
        
        $lead = DB::table('leads')
            ->where('id', $request->lead_id)
            ->whereIn('equipe_id', $equipeIds)
            ->first();
        
        if (!$lead) {
            return back()->withErrors(['error' => 'Lead não encontrado ou não pertence às suas equipes.']);
        }
        
        // Verificar se o especialista pertence às equipes do coordenador
        $especialista = DB::table('equipe_especialistas as ee')
            ->join('equipes as e', 'ee.equipe_id', '=', 'e.id')
            ->where('ee.especialista_id', $request->especialista_id)
            ->where('e.coordenador_id', $coordenador->id)
            ->where('ee.status', 'ativo')
            ->first();
        
        if (!$especialista) {
            return back()->withErrors(['error' => 'Especialista não pertence às suas equipes.']);
        }
        
        // Distribuir lead
        DB::table('leads')
            ->where('id', $request->lead_id)
            ->update([
                'especialista_id' => $request->especialista_id,
                'etapa' => 'contato',
                'data_distribuicao' => now(),
            ]);
        
        return back()->with('success', 'Lead distribuído manualmente com sucesso!');
    }
}

