<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class RelatorioController extends Controller
{
    /**
     * Exibe a página de relatórios consolidados
     */
    public function index(Request $request)
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Filtros
        $mes = $request->get('mes', date('m'));
        $ano = $request->get('ano', date('Y'));
        $especialistaId = $request->get('especialista_id', '');
        $tipoRelatorio = $request->get('tipo', 'geral');
        
        // Estatísticas gerais da equipe
        $statsGerais = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->selectRaw('
                COUNT(DISTINCT a.id) as total_alunos,
                SUM(CASE WHEN a.status = "ativo" THEN 1 ELSE 0 END) as alunos_ativos,
                SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) as alunos_concluidos,
                SUM(CASE WHEN a.status = "trancado" THEN 1 ELSE 0 END) as alunos_parados,
                SUM(CASE WHEN MONTH(a.data_cadastro) = ' . (int)$mes . ' AND YEAR(a.data_cadastro) = ' . (int)$ano . ' THEN 1 ELSE 0 END) as vendas_mes,
                0 as valor_medio_curso,
                0 as faturamento_total
            ')
            ->first();
        
        // Query base para relatório por especialista
        $query = DB::table('usuarios as u')
            ->join('equipe_especialistas as ee', 'u.id', '=', 'ee.especialista_id')
            ->leftJoin('alunos as a', 'u.id', '=', 'a.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo');
        
        if ($especialistaId) {
            $query->where('a.especialista_id', $especialistaId);
        }
        
        // Relatório por especialista
        $relatorioEspecialistas = $query->select(
                'u.id',
                'u.nome',
                DB::raw('COUNT(a.id) as total_alunos'),
                DB::raw('SUM(CASE WHEN MONTH(a.data_cadastro) = ' . (int)$mes . ' AND YEAR(a.data_cadastro) = ' . (int)$ano . ' THEN 1 ELSE 0 END) as vendas_mes'),
                DB::raw('SUM(CASE WHEN a.status = "ativo" THEN 1 ELSE 0 END) as alunos_ativos'),
                DB::raw('SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) as alunos_concluidos'),
                DB::raw('SUM(CASE WHEN a.status = "trancado" THEN 1 ELSE 0 END) as alunos_parados'),
                DB::raw('ROUND((SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) / NULLIF(COUNT(a.id), 0)) * 100, 1) as taxa_conclusao'),
                DB::raw('0 as valor_medio_curso'),
                DB::raw('0 as faturamento_total')
            )
            ->groupBy('u.id', 'u.nome')
            ->orderByDesc('vendas_mes')
            ->orderByDesc('total_alunos')
            ->get();
        
        // Relatório de evolução dos alunos
        $relatorioAlunos = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->join('usuarios as u', 'a.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->select(
                'a.id',
                'a.nome',
                'a.email',
                'a.telefone',
                'a.data_cadastro',
                'a.status',
                'a.modulo_atual',
                'a.progresso_geral as progresso',
                'u.nome as especialista_nome'
            )
            ->orderBy('a.data_cadastro', 'desc')
            ->get();
        
        // Relatório de vendas por período
        $relatorioVendasPeriodo = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('a.data_cadastro', '>=', DB::raw('DATE_SUB(CURRENT_DATE(), INTERVAL 12 MONTH)'))
            ->selectRaw('
                DATE_FORMAT(a.data_cadastro, "%Y-%m") as mes,
                COUNT(*) as vendas,
                0 as faturamento,
                SUM(CASE WHEN a.status = "ativo" THEN 1 ELSE 0 END) as alunos_ativos,
                SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) as alunos_concluidos
            ')
            ->groupBy('mes')
            ->orderBy('mes')
            ->get();
        
        // Especialistas para filtro
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();
        
        return view('coordenador.relatorios.index', compact(
            'equipes',
            'statsGerais',
            'relatorioEspecialistas',
            'relatorioAlunos',
            'relatorioVendasPeriodo',
            'especialistas',
            'mes',
            'ano',
            'especialistaId',
            'tipoRelatorio'
        ));
    }
}

