<?php

namespace App\Http\Controllers\Especialista;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AtendimentoController extends Controller
{
    /**
     * Kanban de Atendimentos - Visão Especialista
     */
    public function index(Request $request)
    {
        $especialista = Auth::user();

        // Buscar leads atribuídos ao especialista
        $leads = DB::table('leads as l')
            ->leftJoin('negociacoes as n', function($join) {
                $join->on('l.id', '=', 'n.lead_id')
                     ->where('n.status', '=', 'fechado')
                     ->whereNotNull('n.aprovado_por');
            })
            ->where('l.especialista_id', $especialista->id)
            ->select(
                'l.id',
                'l.nome',
                'l.email',
                'l.telefone',
                'l.profissao',
                'l.etapa',
                'l.status',
                'l.especialista_id',
                'l.data_cadastro',
                'l.data_distribuicao',
                'l.data_atualizacao',
                'l.observacoes',
                'l.horario_preferencia',
                'l.area_interesse',
                'l.data_agendamento',
                'l.horario_agendamento',
                DB::raw("CASE 
                    WHEN l.etapa = 'novo' THEN 'aguardando'
                    WHEN l.etapa = 'contato' THEN 'contatado'
                    WHEN l.etapa = 'demo' THEN 'negociacao'
                    WHEN l.etapa = 'fechamento' THEN 'aprovada'
                    WHEN l.etapa = 'perdido' THEN 'rejeitada'
                    ELSE 'aguardando'
                END as status_kanban"),
                DB::raw("1 as distribuido_para_mim"),
                DB::raw("'lead' as tipo"),
                DB::raw("CASE WHEN n.id IS NOT NULL THEN 1 ELSE 0 END as venda_aprovada")
            )
            ->orderByRaw("
                CASE 
                    WHEN l.etapa = 'novo' THEN 1
                    WHEN l.etapa = 'contato' THEN 2
                    WHEN l.etapa = 'demo' THEN 3
                    WHEN l.etapa = 'fechamento' THEN 4
                    ELSE 5
                END
            ")
            ->orderBy('l.data_atualizacao', 'desc')
            ->orderBy('l.data_cadastro', 'desc')
            ->get();

        // Buscar leads não atribuídos (disponíveis)
        $leadsDisponiveis = DB::table('leads as l')
            ->leftJoin('negociacoes as n', function($join) {
                $join->on('l.id', '=', 'n.lead_id')
                     ->where('n.status', '=', 'fechado')
                     ->whereNotNull('n.aprovado_por');
            })
            ->whereNull('l.especialista_id')
            ->where('l.etapa', 'novo')
            ->select(
                'l.id',
                'l.nome',
                'l.email',
                'l.telefone',
                'l.profissao',
                'l.etapa',
                'l.status',
                'l.especialista_id',
                'l.data_cadastro',
                'l.data_distribuicao',
                'l.data_atualizacao',
                'l.observacoes',
                'l.horario_preferencia',
                'l.area_interesse',
                'l.data_agendamento',
                'l.horario_agendamento',
                DB::raw("'aguardando' as status_kanban"),
                DB::raw("0 as distribuido_para_mim"),
                DB::raw("'lead' as tipo"),
                DB::raw("CASE WHEN n.id IS NOT NULL THEN 1 ELSE 0 END as venda_aprovada")
            )
            ->orderBy('l.data_cadastro', 'desc')
            ->get();

        // Combinar leads
        $leadsDb = $leads->merge($leadsDisponiveis)->toArray();

        // Estatísticas
        $stats = DB::table('leads')
            ->where('especialista_id', $especialista->id)
            ->select(
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(CASE WHEN etapa = "novo" THEN 1 ELSE 0 END) as aguardando'),
                DB::raw('SUM(CASE WHEN etapa = "contato" THEN 1 ELSE 0 END) as contatado'),
                DB::raw('SUM(CASE WHEN etapa = "demo" THEN 1 ELSE 0 END) as negociacao'),
                DB::raw('SUM(CASE WHEN etapa = "fechamento" THEN 1 ELSE 0 END) as aprovada'),
                DB::raw('SUM(CASE WHEN etapa = "perdido" THEN 1 ELSE 0 END) as rejeitada')
            )
            ->first();

        return view('especialista.atendimentos.index', compact('leadsDb', 'stats'));
    }

    /**
     * Atualizar status do lead via AJAX (Drag and Drop)
     */
    public function atualizarStatus(Request $request, $id)
    {
        $especialista = Auth::user();

        // Primeiro, verificar se o lead existe
        $leadExists = DB::table('leads')->where('id', $id)->first();
        
        if (!$leadExists) {
            return response()->json([
                'success' => false,
                'message' => 'Lead não encontrado',
            ], 404);
        }

        // Buscar lead - pode ser atribuído ao especialista ou não atribuído (disponível)
        $lead = DB::table('leads')
            ->where('id', $id)
            ->where(function($query) use ($especialista) {
                $query->where('especialista_id', $especialista->id)
                      ->orWhereNull('especialista_id');
            })
            ->first();

        if (!$lead) {
            // Se o lead existe mas não está disponível para este especialista
            return response()->json([
                'success' => false,
                'message' => 'Este lead não está disponível para você. Ele pode estar atribuído a outro especialista.',
            ], 403);
        }

        // Normalizar dados antes da validação
        $data = $request->all();
        
        // Garantir que status seja string válida
        if (!isset($data['status']) || $data['status'] === '' || $data['status'] === null) {
            return response()->json([
                'success' => false,
                'message' => 'Status é obrigatório.',
                'errors' => ['status' => ['Status é obrigatório.']]
            ], 422);
        }
        
        $data['status'] = trim((string) $data['status']);
        
        // Garantir que motivo seja string ou null
        if (isset($data['motivo']) && $data['motivo'] !== null && $data['motivo'] !== '') {
            $data['motivo'] = trim((string) $data['motivo']);
        } else {
            $data['motivo'] = null;
        }
        
        // Validar
        $validator = \Validator::make($data, [
            'status' => 'required|string|in:aguardando,contatado,negociacao,aprovada,rejeitada',
            'motivo' => 'nullable|string|max:1000',
        ], [
            'status.required' => 'O status é obrigatório.',
            'status.string' => 'O status deve ser uma string.',
            'status.in' => 'Status inválido. Valores permitidos: aguardando, contatado, negociacao, aprovada, rejeitada.',
            'motivo.string' => 'O motivo deve ser uma string.',
            'motivo.max' => 'O motivo não pode ter mais de 1000 caracteres.',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erro de validação: ' . implode(', ', $validator->errors()->all()),
                'errors' => $validator->errors(),
                'debug' => [
                    'status_received' => $request->input('status'),
                    'status_type' => gettype($request->input('status')),
                    'status_normalized' => $data['status'] ?? 'N/A',
                ]
            ], 422);
        }
        
        // Usar dados normalizados
        $request->merge($data);

        // Mapear status do kanban para etapa do banco
        $etapaMap = [
            'aguardando' => 'novo',
            'contatado' => 'contato',
            'negociacao' => 'demo',
            'aprovada' => 'fechamento',
            'rejeitada' => 'perdido',
        ];

        $etapa = $etapaMap[$request->status] ?? 'novo';

        // Preparar dados para atualização (usar timezone de Brasília)
        $now = Carbon::now('America/Sao_Paulo');
        
        $updateData = [
            'etapa' => $etapa,
            'data_atualizacao' => $now,
        ];

        // Se o lead não estiver atribuído, atribuir ao especialista
        if (!$lead->especialista_id) {
            $updateData['especialista_id'] = $especialista->id;
            $updateData['data_distribuicao'] = $now;
        }

        // Se for "perdido" e houver motivo, adicionar às observações
        if ($etapa === 'perdido' && $request->motivo) {
            $observacoesAtuais = $lead->observacoes ?? '';
            // Formatar data no padrão brasileiro (dd/mm/yyyy HH:mm)
            $novaObservacao = "\n\n[MOTIVO DA PERDA - " . $now->format('d/m/Y H:i') . "]\n" . $request->motivo;
            
            // Concatenar com observações existentes
            $updateData['observacoes'] = trim($observacoesAtuais . $novaObservacao);
        }

        // Se for "aprovada" (fechamento), criar negociação aguardando aprovação
        if ($etapa === 'fechamento') {
            // Verificar se já existe uma negociação para este lead
            $negociacaoExistente = DB::table('negociacoes')
                ->where('lead_id', $id)
                ->where('status', 'andamento')
                ->first();

            if (!$negociacaoExistente) {
                // Criar negociação básica aguardando aprovação
                // O valor será preenchido posteriormente ou pode vir do request
                $valorProposta = $request->valor_proposta ?? 0;
                
                DB::table('negociacoes')->insert([
                    'lead_id' => $id,
                    'especialista_id' => $especialista->id,
                    'valor_proposta' => $valorProposta,
                    'valor_total' => $valorProposta,
                    'status' => 'andamento', // Status 'andamento' indica aguardando aprovação
                    'data_criacao' => $now,
                    'criado_por' => $especialista->id,
                ]);
            }
        }
        
        // Se for "negociacao" (demo), NÃO atualizar ainda - o modal vai fazer isso
        // O frontend vai abrir o modal e só depois de criar a negociação é que atualiza o status
        if ($etapa === 'demo') {
            // Retornar sucesso mas indicar que precisa abrir modal
            return response()->json([
                'success' => true,
                'message' => 'Abrindo modal de negociação...',
                'open_modal' => true,
                'lead_id' => $id,
            ]);
        }

        DB::table('leads')
            ->where('id', $id)
            ->update($updateData);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado!',
            'lead_nome' => $lead->nome,
        ]);
    }
}

