<?php

namespace App\Http\Controllers\Especialista;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ComissaoController extends Controller
{
    /**
     * Listar comissões do especialista
     */
    public function index(Request $request)
    {
        $especialista = Auth::user();

        // Filtros
        $filtroStatus = $request->input('status', 'todas');
        $filtroPeriodo = $request->input('periodo', 'mes');

        // Definir período
        switch ($filtroPeriodo) {
            case 'semana':
                $dataInicio = Carbon::now()->startOfWeek()->format('Y-m-d');
                break;
            case 'trimestre':
                $mesAtual = Carbon::now()->month;
                $trimestreInicio = floor(($mesAtual - 1) / 3) * 3 + 1;
                $dataInicio = Carbon::now()->month($trimestreInicio)->startOfMonth()->format('Y-m-d');
                break;
            case 'ano':
                $dataInicio = Carbon::now()->startOfYear()->format('Y-m-d');
                break;
            case 'mes':
            default:
                $dataInicio = Carbon::now()->startOfMonth()->format('Y-m-d');
                break;
        }

        // Buscar comissões
        $whereClauses = ["c.especialista_id = ?", "n.data_criacao >= ?"];
        $params = [$especialista->id, $dataInicio];

        if ($filtroStatus !== 'todas') {
            $whereClauses[] = "c.status = ?";
            $params[] = $filtroStatus;
        }

        $whereSql = implode(' AND ', $whereClauses);

        $comissoes = DB::select("
            SELECT 
                c.*,
                n.valor_total as valor_venda,
                n.num_parcelas,
                n.data_criacao as data_negociacao,
                a.nome as aluno_nome,
                a.email as aluno_email,
                l.nome as lead_nome
            FROM comissoes c
            INNER JOIN negociacoes n ON c.negociacao_id = n.id
            INNER JOIN alunos a ON n.aluno_id = a.id
            LEFT JOIN vinculo v ON n.aluno_id = v.aluno_id AND v.status = 'ativo'
            LEFT JOIN leads l ON v.lead_id = l.id
            WHERE {$whereSql}
            ORDER BY n.data_criacao DESC
        ", $params);

        // Converter para collection
        $comissoes = collect($comissoes);

        // Estatísticas
        $stats = [
            'total_comissoes' => $comissoes->count(),
            'valor_pendente' => $comissoes->where('status', 'pendente')->sum('valor'),
            'valor_pago' => $comissoes->whereIn('status', ['paga', 'liberada'])->sum('valor'),
            'valor_total' => $comissoes->sum('valor'),
            'qtd_pendente' => $comissoes->where('status', 'pendente')->count(),
            'qtd_paga' => $comissoes->whereIn('status', ['paga', 'liberada'])->count(),
        ];

        // Comissões por mês (últimos 6 meses) - usando data da negociação
        $comissoesPorMes = DB::select("
            SELECT 
                DATE_FORMAT(n.data_criacao, '%Y-%m') as mes,
                COUNT(*) as quantidade,
                SUM(c.valor) as valor_total,
                SUM(CASE WHEN c.status IN ('paga', 'liberada') THEN c.valor ELSE 0 END) as valor_pago
            FROM comissoes c
            INNER JOIN negociacoes n ON c.negociacao_id = n.id
            WHERE c.especialista_id = ?
            AND n.data_criacao >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
            GROUP BY DATE_FORMAT(n.data_criacao, '%Y-%m')
            ORDER BY mes
        ", [$especialista->id]);

        return view('especialista.comissoes.index', compact(
            'comissoes',
            'stats',
            'comissoesPorMes',
            'filtroStatus',
            'filtroPeriodo'
        ));
    }
}

