<?php

namespace App\Http\Controllers\Especialista;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Venda;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Exibe o dashboard do especialista
     */
    public function index(Request $request)
    {
        $usuario = Auth::user();

        if ($usuario->role !== 'especialista') {
            abort(403, 'Acesso negado.');
        }

        // Parâmetros de período
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // ============================================
        // KPIs DO ESPECIALISTA
        // ============================================

        // Leads ativos
        $leadsAtivos = Lead::where('especialista_id', $usuario->id)
            ->whereIn('etapa', ['novo', 'contato', 'demo', 'fechamento'])
            ->count();

        // Leads do período
        $leadsPeriodo = Lead::where('especialista_id', $usuario->id)
            ->whereBetween(DB::raw('DATE(data_distribuicao)'), [$start, $end])
            ->count();

        // Matrículas do período (através de alunos)
        $vendasPeriodo = DB::table('matriculas as m')
            ->join('alunos as a', 'm.aluno_id', '=', 'a.id')
            ->where('a.especialista_id', $usuario->id)
            ->whereBetween(DB::raw('DATE(m.data_matricula)'), [$start, $end])
            ->where('m.status', 'ativa')
            ->count();

        // Receita do período (negociações fechadas)
        $receitaPeriodo = DB::table('negociacoes')
            ->where('especialista_id', $usuario->id)
            ->whereBetween(DB::raw('DATE(data_fechamento)'), [$start, $end])
            ->where('status', 'fechado')
            ->sum(DB::raw('COALESCE(valor_fechado, valor_total, 0)')) ?? 0;

        // Taxa de conversão
        $taxaConversao = $leadsPeriodo > 0 ? ($vendasPeriodo / $leadsPeriodo) * 100 : 0;

        // Negociações pendentes de aprovação
        $vendasPendentes = DB::table('negociacoes')
            ->where('especialista_id', $usuario->id)
            ->whereIn('status', ['pendente', 'em_analise'])
            ->count();

        // ============================================
        // LEADS POR STATUS (KANBAN)
        // ============================================
        $leadsKanban = [
            'novo' => Lead::where('especialista_id', $usuario->id)
                ->where('etapa', 'novo')
                ->orderBy('data_distribuicao', 'desc')
                ->limit(10)
                ->get(),
            
            'contato' => Lead::where('especialista_id', $usuario->id)
                ->where('etapa', 'contato')
                ->orderBy('data_atualizacao', 'desc')
                ->limit(10)
                ->get(),
            
            'demo' => Lead::where('especialista_id', $usuario->id)
                ->where('etapa', 'demo')
                ->orderBy('data_atualizacao', 'desc')
                ->limit(10)
                ->get(),
            
            'fechamento' => Lead::where('especialista_id', $usuario->id)
                ->where('etapa', 'fechamento')
                ->orderBy('data_atualizacao', 'desc')
                ->limit(10)
                ->get(),
        ];

        // ============================================
        // PRÓXIMOS ATENDIMENTOS (Leads com agendamento)
        // ============================================
        $proximosAtendimentos = Lead::where('especialista_id', $usuario->id)
            ->whereNotNull('data_agendamento')
            ->whereDate('data_agendamento', '>=', now()->format('Y-m-d'))
            ->orderBy('data_agendamento')
            ->orderBy('horario_agendamento')
            ->limit(5)
            ->get();

        // ============================================
        // EVOLUÇÃO DAS MATRÍCULAS
        // ============================================
        $evolucaoMatriculas = DB::table('matriculas as m')
            ->join('alunos as a', 'm.aluno_id', '=', 'a.id')
            ->where('a.especialista_id', $usuario->id)
            ->whereBetween(DB::raw('DATE(m.data_matricula)'), [$start, $end])
            ->where('m.status', 'ativa')
            ->selectRaw('DATE(m.data_matricula) as data, COUNT(*) as total')
            ->groupBy('data')
            ->orderBy('data')
            ->get();

        // ============================================
        // META MENSAL
        // ============================================
        $metaMensal = 0; // Pode buscar de uma tabela de metas se existir
        $progressoMeta = $metaMensal > 0 ? ($vendasPeriodo / $metaMensal) * 100 : 0;

        return view('especialista.dashboard', compact(
            'start',
            'end',
            'leadsAtivos',
            'leadsPeriodo',
            'vendasPeriodo',
            'receitaPeriodo',
            'taxaConversao',
            'vendasPendentes',
            'proximosAtendimentos',
            'evolucaoMatriculas',
            'metaMensal',
            'progressoMeta'
        ));
    }
}

