<?php

namespace App\Http\Controllers\Especialista;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Venda;
use App\Models\Atendimento;
use App\Models\Voucher;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class NegociacaoController extends Controller
{
    /**
     * Lista leads do especialista
     */
    public function index(Request $request)
    {
        $usuario = Auth::user();

        $query = Lead::where('especialista_id', $usuario->id);

        // Filtros
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nome', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('telefone', 'like', "%{$search}%");
            });
        }

        $leads = $query->orderBy('data_distribuicao', 'desc')->paginate(50);

        return view('especialista.negociacao.index', compact('leads'));
    }

    /**
     * Exibe detalhes de um lead
     */
    public function show($id)
    {
        $usuario = Auth::user();

        $lead = Lead::where('especialista_id', $usuario->id)
            ->with('atendimentos')
            ->findOrFail($id);

        return view('especialista.negociacao.show', compact('lead'));
    }

    /**
     * Atualiza status do lead
     */
    public function atualizarStatus(Request $request, $id)
    {
        $usuario = Auth::user();

        $request->validate([
            'status' => 'required|in:em_contato,negociacao,proposta_enviada,perdido',
            'observacoes' => 'nullable|string',
            'motivo_perda' => 'required_if:status,perdido',
        ]);

        $lead = Lead::where('especialista_id', $usuario->id)->findOrFail($id);

        $data = [
            'status' => $request->status,
            'data_ultimo_contato' => now(),
        ];

        if ($request->filled('observacoes')) {
            $data['observacoes'] = ($lead->observacoes ? $lead->observacoes . "\n\n" : '') 
                . "[" . now()->format('d/m/Y H:i') . "] " . $request->observacoes;
        }

        if ($request->status === 'perdido') {
            $data['motivo_perda'] = $request->motivo_perda;
        }

        if ($request->status === 'em_contato' && !$lead->data_primeiro_contato) {
            $data['data_primeiro_contato'] = now();
        }

        $lead->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado com sucesso!',
        ]);
    }

    /**
     * Registra um atendimento
     */
    public function registrarAtendimento(Request $request, $id)
    {
        $usuario = Auth::user();

        $request->validate([
            'tipo' => 'required|in:ligacao,whatsapp,email,presencial',
            'observacoes' => 'required|string',
            'resultado' => 'nullable|in:positivo,negativo,neutro',
        ]);

        $lead = Lead::where('especialista_id', $usuario->id)->findOrFail($id);

        Atendimento::create([
            'lead_id' => $lead->id,
            'especialista_id' => $usuario->id,
            'tipo' => $request->tipo,
            'status' => 'realizado',
            'data_realizacao' => now(),
            'observacoes' => $request->observacoes,
            'resultado' => $request->resultado,
        ]);

        // Atualizar data do último contato
        $lead->update(['data_ultimo_contato' => now()]);

        return response()->json([
            'success' => true,
            'message' => 'Atendimento registrado com sucesso!',
        ]);
    }

    /**
     * Cria uma nova venda
     */
    public function criarVenda(Request $request, $id)
    {
        $usuario = Auth::user();

        $request->validate([
            'valor_total' => 'required|numeric|min:0',
            'valor_entrada' => 'nullable|numeric|min:0',
            'num_parcelas' => 'required|integer|min:1',
            'forma_pagamento' => 'required|in:dinheiro,pix,cartao_credito,cartao_debito,boleto',
            'voucher_codigo' => 'nullable|string',
            'observacoes' => 'nullable|string',
        ]);

        $lead = Lead::where('especialista_id', $usuario->id)->findOrFail($id);

        DB::beginTransaction();
        try {
            $valorTotal = $request->valor_total;
            $voucherId = null;

            // Aplicar voucher se fornecido
            if ($request->filled('voucher_codigo')) {
                $voucher = Voucher::where('codigo', $request->voucher_codigo)
                    ->where('status', 'ativo')
                    ->first();

                if ($voucher && $voucher->isAtivo()) {
                    if ($voucher->tipo === 'percentual') {
                        $desconto = ($valorTotal * $voucher->percentual_desconto) / 100;
                        $valorTotal -= $desconto;
                    } else {
                        $valorTotal -= $voucher->valor_desconto;
                    }

                    $voucherId = $voucher->id;

                    // Marcar voucher como usado
                    $voucher->update([
                        'status' => 'usado',
                        'usado_por' => $usuario->id,
                        'data_uso' => now(),
                    ]);
                }
            }

            $valorEntrada = $request->valor_entrada ?? 0;
            $valorRestante = $valorTotal - $valorEntrada;
            $valorParcela = $request->num_parcelas > 0 ? $valorRestante / $request->num_parcelas : 0;

            // Criar negociação
            $negociacaoId = DB::table('negociacoes')->insertGetId([
                'lead_id' => $lead->id,
                'especialista_id' => $usuario->id,
                'valor_total' => $valorTotal,
                'valor_entrada' => $valorEntrada,
                'num_parcelas' => $request->num_parcelas,
                'valor_parcela' => $valorParcela,
                'forma_pagamento' => $request->forma_pagamento,
                'status' => 'pendente',
                'data_criacao' => now(),
                'observacoes' => $request->observacoes,
                'voucher_id' => $voucherId,
            ]);

            // Atualizar status do lead
            $lead->update(['status' => 'proposta_enviada']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Negociação criada com sucesso! Aguardando aprovação.',
                'negociacao_id' => $negociacaoId,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Erro ao criar venda: ' . $e->getMessage(),
            ], 500);
        }
    }
}

