<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Comissao extends Model
{
    use HasFactory;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'comissoes';

    /**
     * Desabilitar timestamps automáticos do Laravel
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'negociacao_id',
        'especialista_id',
        'valor',
        'percentual',
        'status',
        'data_prevista',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'valor' => 'decimal:2',
        'percentual' => 'decimal:2',
        'data_prevista' => 'date',
    ];

    /**
     * Relacionamento com Negociacao
     */
    public function negociacao()
    {
        return $this->belongsTo(Negociacao::class, 'negociacao_id');
    }

    /**
     * Relacionamento com Especialista (Usuario)
     */
    public function especialista()
    {
        return $this->belongsTo(Usuario::class, 'especialista_id');
    }

    /**
     * Verifica se a comissão está pendente
     */
    public function isPendente(): bool
    {
        return $this->status === 'pendente';
    }

    /**
     * Verifica se a comissão está liberada
     */
    public function isLiberada(): bool
    {
        return $this->status === 'liberada';
    }

    /**
     * Verifica se a comissão está paga
     */
    public function isPaga(): bool
    {
        return $this->status === 'paga';
    }

    /**
     * Scope para comissões pendentes
     */
    public function scopePendentes($query)
    {
        return $query->where('status', 'pendente');
    }

    /**
     * Scope para comissões liberadas
     */
    public function scopeLiberadas($query)
    {
        return $query->where('status', 'liberada');
    }

    /**
     * Scope para comissões pagas
     */
    public function scopePagas($query)
    {
        return $query->where('status', 'paga');
    }
}

