<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Especialista extends Model
{
    use HasFactory;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'especialistas';

    /**
     * Desabilitar timestamps automáticos do Laravel
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'usuario_id',
        'equipe_id',
        'status',
        'meta_mensal',
        'comissao_percentual',
        'data_admissao',
        'data_desligamento',
        'observacoes',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data_admissao' => 'date',
        'data_desligamento' => 'date',
        'meta_mensal' => 'decimal:2',
        'comissao_percentual' => 'decimal:2',
    ];

    /**
     * Relacionamento com Usuario
     */
    public function usuario()
    {
        return $this->belongsTo(Usuario::class, 'usuario_id');
    }

    /**
     * Relacionamento com Equipe
     */
    public function equipe()
    {
        return $this->belongsTo(Equipe::class, 'equipe_id');
    }

    /**
     * Relacionamento com Leads
     */
    public function leads()
    {
        return $this->hasMany(Lead::class, 'especialista_id');
    }

    /**
     * Relacionamento com Vendas
     */
    public function vendas()
    {
        return $this->hasMany(Venda::class, 'especialista_id');
    }

    /**
     * Relacionamento com Atendimentos
     */
    public function atendimentos()
    {
        return $this->hasMany(Atendimento::class, 'especialista_id');
    }

    /**
     * Verifica se o especialista está ativo
     */
    public function isAtivo(): bool
    {
        return $this->status === 'ativo';
    }

    /**
     * Scope para especialistas ativos
     */
    public function scopeAtivos($query)
    {
        return $query->where('status', 'ativo');
    }

    /**
     * Scope para especialistas por equipe
     */
    public function scopePorEquipe($query, $equipeId)
    {
        return $query->where('equipe_id', $equipeId);
    }
}

