<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Lead extends Model
{
    use HasFactory;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'leads';

    /**
     * Desabilitar timestamps automáticos do Laravel
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'nome',
        'email',
        'telefone',
        'profissao',
        'horario_preferencia',
        'area_interesse',
        'origem',
        'etapa',
        'status',
        'especialista_id',
        'equipe_id',
        'link_marketing_id',
        'coordenador_id',
        'data_cadastro',
        'data_distribuicao',
        'data_captura',
        'data_atualizacao',
        'data_agendamento',
        'horario_agendamento',
        'observacoes',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data_cadastro' => 'datetime',
        'data_distribuicao' => 'datetime',
        'data_captura' => 'datetime',
        'data_atualizacao' => 'datetime',
        'data_agendamento' => 'date',
    ];

    /**
     * Relacionamento com Especialista (Usuario)
     */
    public function especialista()
    {
        return $this->belongsTo(Usuario::class, 'especialista_id');
    }

    /**
     * Relacionamento com Equipe
     */
    public function equipe()
    {
        return $this->belongsTo(Equipe::class, 'equipe_id');
    }

    /**
     * Relacionamento com Campanha
     */
    public function campanha()
    {
        return $this->belongsTo(Campanha::class, 'campanha_id');
    }

    /**
     * Relacionamento com Atendimentos
     */
    public function atendimentos()
    {
        return $this->hasMany(Atendimento::class, 'lead_id');
    }

    /**
     * Relacionamento com Vendas
     */
    public function vendas()
    {
        return $this->hasMany(Venda::class, 'lead_id');
    }

    /**
     * Verifica se o lead está ativo
     */
    public function isAtivo(): bool
    {
        return $this->status === 'ativo';
    }

    /**
     * Verifica se o lead está em processo
     */
    public function isEmProcesso(): bool
    {
        return in_array($this->etapa, ['novo', 'contato', 'demo', 'fechamento']);
    }

    /**
     * Verifica se o lead foi perdido
     */
    public function isPerdido(): bool
    {
        return $this->etapa === 'perdido';
    }

    /**
     * Scope para leads ativos
     */
    public function scopeAtivos($query)
    {
        return $query->where('status', 'ativo');
    }

    /**
     * Scope para leads em processo
     */
    public function scopeEmProcesso($query)
    {
        return $query->whereIn('etapa', ['novo', 'contato', 'demo', 'fechamento']);
    }

    /**
     * Scope para leads perdidos
     */
    public function scopePerdidos($query)
    {
        return $query->where('etapa', 'perdido');
    }

    /**
     * Scope para leads por especialista
     */
    public function scopePorEspecialista($query, $especialistaId)
    {
        return $query->where('especialista_id', $especialistaId);
    }

    /**
     * Scope para leads por equipe
     */
    public function scopePorEquipe($query, $equipeId)
    {
        return $query->where('equipe_id', $equipeId);
    }
}

