<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class Usuario extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'usuarios';

    /**
     * Desabilitar timestamps automáticos do Laravel
     * O banco usa criado_em e atualizado_em
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'nome',
        'email',
        'senha',
        'remember_token',
        'telefone',
        'status',
        'role',
        'data_cadastro',
        'ultimo_login',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'senha',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data_cadastro' => 'datetime',
        'ultimo_login' => 'datetime',
    ];

    /**
     * Get the password for authentication.
     */
    public function getAuthPassword()
    {
        return $this->senha;
    }

    /**
     * Verifica se o usuário está ativo
     */
    public function isAtivo(): bool
    {
        return $this->status === 'ativo';
    }

    /**
     * Verifica se o usuário é admin
     */
    public function isAdmin(): bool
    {
        return $this->role === 'admin';
    }

    /**
     * Verifica se o usuário é especialista
     */
    public function isEspecialista(): bool
    {
        return $this->role === 'especialista';
    }

    /**
     * Verifica se o usuário é coordenador
     */
    public function isCoordenador(): bool
    {
        return $this->role === 'coordenador';
    }

    /**
     * Verifica se o usuário é secretaria
     */
    public function isSecretaria(): bool
    {
        return $this->role === 'secretaria';
    }

    /**
     * Verifica se o usuário é professor
     */
    public function isProfessor(): bool
    {
        return $this->role === 'professor';
    }

    /**
     * Verifica se o usuário é aluno
     */
    public function isAluno(): bool
    {
        return $this->role === 'aluno';
    }

    /**
     * Relacionamento com Aluno
     */
    public function aluno()
    {
        return $this->hasOne(Aluno::class, 'usuario_id');
    }

    /**
     * Relacionamento com Equipes (se for especialista)
     */
    public function equipes()
    {
        return $this->belongsToMany(
            Equipe::class,
            'equipe_especialistas',
            'especialista_id',
            'equipe_id'
        )->wherePivot('status', 'ativo');
    }

    /**
     * Relacionamento com Leads (se for especialista)
     */
    public function leadsEspecialista()
    {
        return $this->hasMany(Lead::class, 'especialista_id');
    }
}

