<?php

namespace App\Services;

use App\Models\Lead;
use App\Models\Negociacao;
use App\Models\Usuario;
use App\Models\Equipe;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RelatorioService
{
    /**
     * Gera relatório consolidado de vendas
     */
    public function relatorioVendas(string $dataInicio, string $dataFim, ?int $equipeId = null, ?int $especialistaId = null): array
    {
        $query = Negociacao::where('status', 'fechado')
            ->whereBetween(DB::raw('DATE(data_fechamento)'), [$dataInicio, $dataFim]);

        if ($especialistaId) {
            $query->where('especialista_id', $especialistaId);
        } elseif ($equipeId) {
            // Buscar IDs dos especialistas da equipe
            $especialistaIds = DB::table('equipe_especialistas')
                ->where('equipe_id', $equipeId)
                ->where('status', 'ativo')
                ->pluck('especialista_id')
                ->toArray();
            
            if (!empty($especialistaIds)) {
                $query->whereIn('especialista_id', $especialistaIds);
            } else {
                // Se não houver especialistas na equipe, retornar vazio
                $query->whereRaw('1 = 0');
            }
        }

        $vendas = $query->with(['lead', 'especialista'])->get();

        $totalVendas = $vendas->count();
        $receitaTotal = $vendas->sum(function($v) {
            return $v->valor_fechado ?? $v->valor_total ?? 0;
        });
        $ticketMedio = $totalVendas > 0 ? $receitaTotal / $totalVendas : 0;

        // Vendas por especialista
        $porEspecialista = $vendas->groupBy('especialista_id')->map(function ($vendasEsp) {
            $especialista = $vendasEsp->first()->especialista;
            
            return [
                'especialista' => $especialista ? $especialista->nome : 'Especialista não encontrado',
                'total_vendas' => $vendasEsp->count(),
                'receita' => $vendasEsp->sum(function($v) {
                    return $v->valor_fechado ?? $v->valor_total ?? 0;
                }),
                'ticket_medio' => $vendasEsp->avg(function($v) {
                    return $v->valor_fechado ?? $v->valor_total ?? 0;
                }),
            ];
        })->values();

        // Vendas por dia
        $porDia = $vendas->groupBy(function($venda) {
            $data = $venda->data_fechamento ?? $venda->data_criacao;
            return $data ? Carbon::parse($data)->format('Y-m-d') : date('Y-m-d');
        })->map(function ($vendasDia, $data) {
            return [
                'data' => Carbon::parse($data)->format('d/m/Y'),
                'total_vendas' => $vendasDia->count(),
                'receita' => $vendasDia->sum(function($v) {
                    return $v->valor_fechado ?? $v->valor_total ?? 0;
                }),
            ];
        })->values();

        // Forma de pagamento (se existir)
        $porFormaPagamento = collect([]);
        try {
            $porFormaPagamento = $vendas->groupBy('forma_pagamento')->map(function ($vendasForma, $forma) {
                return [
                    'forma' => $forma ?: 'Não informado',
                    'total' => $vendasForma->count(),
                    'receita' => $vendasForma->sum(function($v) {
                        return $v->valor_fechado ?? $v->valor_total ?? 0;
                    }),
                ];
            })->values();
        } catch (\Exception $e) {
            // Se não houver forma_pagamento, retornar coleção vazia
        }

        return [
            'periodo' => [
                'inicio' => Carbon::parse($dataInicio)->format('d/m/Y'),
                'fim' => Carbon::parse($dataFim)->format('d/m/Y'),
            ],
            'resumo' => [
                'total_vendas' => $totalVendas,
                'receita_total' => $receitaTotal,
                'ticket_medio' => $ticketMedio,
            ],
            'por_especialista' => $porEspecialista,
            'por_dia' => $porDia,
            'por_forma_pagamento' => $porFormaPagamento,
        ];
    }

    /**
     * Gera relatório de performance de especialistas
     */
    public function relatorioPerformance(string $dataInicio, string $dataFim, ?int $equipeId = null): array
    {
        $query = Usuario::where('role', 'especialista')
            ->where('status', 'ativo');

        if ($equipeId) {
            $query->whereHas('equipes', function($q) use ($equipeId) {
                $q->where('equipes.id', $equipeId);
            });
        }

        $especialistas = $query->with('equipes')->get();

        $dados = $especialistas->map(function ($especialista) use ($dataInicio, $dataFim) {
            // Leads distribuídos
            $leadsDistribuidos = Lead::where('especialista_id', $especialista->id)
                ->whereBetween(DB::raw('DATE(data_distribuicao)'), [$dataInicio, $dataFim])
                ->count();

            // Vendas realizadas (negociações fechadas)
            $vendas = Negociacao::where('especialista_id', $especialista->id)
                ->where('status', 'fechado')
                ->whereBetween(DB::raw('DATE(data_fechamento)'), [$dataInicio, $dataFim])
                ->get();

            $totalVendas = $vendas->count();
            $receitaGerada = $vendas->sum(function($v) {
                return $v->valor_fechado ?? $v->valor_total ?? 0;
            });
            $taxaConversao = $leadsDistribuidos > 0 ? ($totalVendas / $leadsDistribuidos) * 100 : 0;

            // Leads perdidos
            $leadsPerdidos = Lead::where('especialista_id', $especialista->id)
                ->where('etapa', 'perdido')
                ->whereBetween(DB::raw('DATE(data_cadastro)'), [$dataInicio, $dataFim])
                ->count();

            // Buscar equipe do especialista
            $equipe = $especialista->equipes->first();
            $equipeNome = $equipe ? $equipe->nome : '-';

            // Meta mensal (buscar da tabela pivot ou usar 0)
            $metaMensal = 0;
            if ($equipe) {
                $pivot = DB::table('equipe_especialistas')
                    ->where('equipe_id', $equipe->id)
                    ->where('especialista_id', $especialista->id)
                    ->first();
                // Se não houver meta na pivot, usar meta da equipe
                $metaMensal = $equipe->meta_mensal ?? 0;
            }

            return [
                'especialista_id' => $especialista->id,
                'nome' => $especialista->nome,
                'equipe' => $equipeNome,
                'leads_distribuidos' => $leadsDistribuidos,
                'vendas_realizadas' => $totalVendas,
                'receita_gerada' => $receitaGerada,
                'taxa_conversao' => round($taxaConversao, 2),
                'leads_perdidos' => $leadsPerdidos,
                'ticket_medio' => $totalVendas > 0 ? $receitaGerada / $totalVendas : 0,
                'meta_mensal' => $metaMensal,
                'atingimento_meta' => $metaMensal > 0 
                    ? round(($totalVendas / $metaMensal) * 100, 2) 
                    : 0,
            ];
        });

        // Ordenar por vendas realizadas
        $dados = $dados->sortByDesc('vendas_realizadas')->values();

        return [
            'periodo' => [
                'inicio' => Carbon::parse($dataInicio)->format('d/m/Y'),
                'fim' => Carbon::parse($dataFim)->format('d/m/Y'),
            ],
            'total_especialistas' => $dados->count(),
            'resumo' => [
                'total_leads' => $dados->sum('leads_distribuidos'),
                'total_vendas' => $dados->sum('vendas_realizadas'),
                'receita_total' => $dados->sum('receita_gerada'),
                'taxa_conversao_media' => $dados->avg('taxa_conversao'),
            ],
            'especialistas' => $dados,
        ];
    }

    /**
     * Gera relatório de funil de vendas
     */
    public function relatorioFunil(string $dataInicio, string $dataFim, ?int $equipeId = null): array
    {
        $query = Lead::whereBetween(DB::raw('DATE(data_cadastro)'), [$dataInicio, $dataFim]);

        if ($equipeId) {
            $query->where('equipe_id', $equipeId);
        }

        $leads = $query->get();

        $totalLeads = $leads->count();

        $funil = [
            'novo' => $leads->where('etapa', 'novo')->count(),
            'em_contato' => $leads->where('etapa', 'contato')->count(),
            'negociacao' => $leads->where('etapa', 'demo')->count(),
            'proposta_enviada' => 0, // Não existe na estrutura atual
            'convertido' => $leads->where('etapa', 'fechamento')->count(),
            'perdido' => $leads->where('etapa', 'perdido')->count(),
        ];

        // Calcular percentuais
        $funilPercentual = [];
        foreach ($funil as $status => $quantidade) {
            $funilPercentual[$status] = [
                'quantidade' => $quantidade,
                'percentual' => $totalLeads > 0 ? round(($quantidade / $totalLeads) * 100, 2) : 0,
            ];
        }

        // Motivos de perda (se existir a coluna motivo_perda)
        $motivosPerda = collect([]);
        try {
            $motivosPerda = Lead::where('etapa', 'perdido')
                ->whereBetween(DB::raw('DATE(data_cadastro)'), [$dataInicio, $dataFim])
                ->when($equipeId, function($q) use ($equipeId) {
                    $q->where('equipe_id', $equipeId);
                })
                ->select('motivo_perda', DB::raw('COUNT(*) as total'))
                ->groupBy('motivo_perda')
                ->get();
        } catch (\Exception $e) {
            // Se a coluna motivo_perda não existir, retornar coleção vazia
        }

        return [
            'periodo' => [
                'inicio' => Carbon::parse($dataInicio)->format('d/m/Y'),
                'fim' => Carbon::parse($dataFim)->format('d/m/Y'),
            ],
            'total_leads' => $totalLeads,
            'funil' => $funilPercentual,
            'taxa_conversao' => $totalLeads > 0 ? round(($funil['convertido'] / $totalLeads) * 100, 2) : 0,
            'taxa_perda' => $totalLeads > 0 ? round(($funil['perdido'] / $totalLeads) * 100, 2) : 0,
            'motivos_perda' => $motivosPerda,
        ];
    }

    /**
     * Exporta relatório para CSV
     */
    public function exportarCSV(array $dados, string $tipo): string
    {
        $filename = storage_path('app/public/relatorios/' . $tipo . '_' . date('Y-m-d_His') . '.csv');
        
        // Criar diretório se não existir
        if (!is_dir(dirname($filename))) {
            mkdir(dirname($filename), 0755, true);
        }

        $handle = fopen($filename, 'w');

        // Adicionar BOM para UTF-8
        fprintf($handle, chr(0xEF).chr(0xBB).chr(0xBF));

        // Cabeçalhos baseados no tipo
        if ($tipo === 'vendas') {
            fputcsv($handle, ['Especialista', 'Total Vendas', 'Receita', 'Ticket Médio']);
            foreach ($dados['por_especialista'] as $linha) {
                fputcsv($handle, [
                    $linha['especialista'],
                    $linha['total_vendas'],
                    number_format($linha['receita'], 2, ',', '.'),
                    number_format($linha['ticket_medio'], 2, ',', '.'),
                ]);
            }
        } elseif ($tipo === 'performance') {
            fputcsv($handle, ['Nome', 'Equipe', 'Leads', 'Vendas', 'Receita', 'Taxa Conversão', 'Ticket Médio']);
            foreach ($dados['especialistas'] as $linha) {
                fputcsv($handle, [
                    $linha['nome'],
                    $linha['equipe'],
                    $linha['leads_distribuidos'],
                    $linha['vendas_realizadas'],
                    number_format($linha['receita_gerada'], 2, ',', '.'),
                    $linha['taxa_conversao'] . '%',
                    number_format($linha['ticket_medio'], 2, ',', '.'),
                ]);
            }
        }

        fclose($handle);

        return $filename;
    }
}

